/************************************************************************
 * \file: trace_base.h
 *
 * \version: $Id: trace_base.h,v 1.62 2012/05/28 10:32:34 jayanth.mc Exp $
 *
 * This file implements the macros, structures for Trace internal use.
 *
 * \component: Gen2 Trace
 *
 * \author B. Das         bhaskar.das(o)in.bosch.com
 *         Sakthivelu S.  sakthivelu.s(o)in.bosch.com
 *
 * \copyright: (c) 2003 - 2009 ADIT
 *
 ***********************************************************************/
#ifndef _TRACE_BASE_H_
#define _TRACE_BASE_H_

#include "trace_impl.h"
#include "trace_q.h"

#ifdef TRACE_ALD_INTERFACE_ENABLED
#include "trace_ald_interface.h"
#endif

#define TR_CHN_EXTN_IDENTIFICATION		0xFF
#define TRACE_EMPTY_STR         ""
#define TRACE_ERR_MEM_BLK_MODE_STR_SIZE 60

typedef ER TRACE_TCPIP_ER;

/* Constant for storing the current version of     *
 * FE/BE which would be stored in to shared memory */
 /*agv2kor*/
# define TRACE_CURRENT_VERSION 4

enum EVT_TYPE{
	EVT_COMMON = 0,
	EVT_TRC_Q,
	EVT_CHNL_Q,
	EVT_PRXY_Q,
	EVT_SND_CMD,
	EVT_SWI_CHNL,
	EVT_MAX
};

typedef union _TRACE_CHNL_PKT
{
  U8*                        pkt;
  TRACE_IN_MULTI_NORM_PKT*   mpnormal;
  TRACE_IN_MULTI_START_PKT*  mpstart;
  TRACE_IN_MULTI_MIDDLE_PKT* mpmiddle;
  TRACE_IN_MULTI_END_PKT*    mpend;
}TRACE_CHNL_PKT;

typedef union _TRACE_PTRS
{
  U8*   p1;
  U32   p2; 
}TRACE_PTRS;


typedef union _TRACE_PROCESS_FRAMES
{
  U8*           pkt;
  TRACE_IN_MSG* msg;
}TRACE_PROCESS_FRAMES;

typedef union _TRACE_UART_WRI
{
  U8*                 buf;
  TRACE_TRACEOUT_MSG* omsg;
}TRACE_UART_WRI;

/* Active communication interface used by Trace */
typedef union
{
  TRACE_uart      dev_uart;
  TRACE_sim       dev_sim;
  TRACE_proxy     dev_pxy;
}TRACE_io_mode;

/* Handle to IO device */
typedef union _TRACE_devid
{
  S32   dev_id;                         /* device ID */
  S32   sock_id;                        /* Socket ID */
}TRACE_devid;

/* Configuration parameters - independent of io device */
typedef struct
{
  TRACE_io_mode       io_dev;         /* io channel used for communication        */
  S32                 io_timeout;     /* timeout for all io operation             */
  U32                 rtry_cnt;       /* max attempt, on any error case           */
  U32                 max_frmsz;      /* max frame size for single transaction    */
  U32                 max_pktcnt;     /* max pkts in a frame                      */
  U32                 min_pktcnt;     /* min pkts in a frame                      */
  U8                  chksum;         /* checksum status (required or not)        */
  TRACE_devid         hdev;           /* Handle to device ID                      */
}TRACE_io_cnfg;


/* Various status flags */
typedef struct
{
  TRACE_TIMESTAMP_MODE timestamp:32; /* Timestamp mode          */  
  TRACE_trip_mode trip:32;           /* Trip mode               */
  U8              blck_mode;         /* Blockmode status        */
  U8              svcfg_onexit;      /* Save cfg on exit or not */
  U8              stat_byte;         /* Status byte update      */
  U8              app_filter;        /* application filter status */
}TRACE_dev_flgs;

/* to do : Cyclic handler based traceout events */
typedef struct
{
  S32      id ;                    /* Handler id             */ 
  U32  cyc_tim;                    /* Time interval          */
}TRACE_cyc_hdlr;

/* to do :Misc. user specified options */
typedef struct
{
  /* Changed from U8 to U32 as a part of Lint Error removal - agv2kor*/
  U32              use_statcls :1;    /* use static class cnfg             */
  U32              use_cychdlr :1;    /* use cyclic handler for buffer o/p */
  TRACE_cyc_hdlr  cyc;
}TRACE_usr_cnfg;

/* Trace device parameters */
typedef struct
{
  TRACE_iodev         actv_chan:32;                 /* Active IO channel            */
  TRACE_iodev         swi_chan:32;                  /* Requested IO channel         */
  TRACE_dev_flgs      flgs;                         /* Status flags                 */
  S8                  lfs_devnm[TRACE_DEVNM_LEN + 1]; /* LFS dev name for cfg files   */
  S8                  chnl_skt_loc[TRACE_DEVNM_LEN + 1]; /*SWGIII-7042 Loc for socket file cretaion*/
  S8                  chnl_skt_ntfy_loc[TRACE_DEVNM_LEN + 1];/*SWGIII-7042loc for socket notify*/
}TRACE_dev_cnfg;

/* Trace callback info */
typedef struct
{
  S32              chn_id;
  pid_t        prc_id;
  TRACE_CALLBACK   fp;
}TRACE_callback_mgr;

/* Trace callback manager - shared info */
typedef struct
{
  TRACE_q_sh   buf; /* Channel Q - shared info */
}TRACE_chnl_sh;

/* Trace channel manager */
typedef  struct
{
  TRACE_chnl_sh*      sh;      /* Pointer to shared info */
  TRACE_q             buf;     /* Channel Q */
  TRACE_callback_mgr* cmgr;    /* Callback manager */  
  U8*                 lrg_pkt; /* Pointer to large packet */
  TRACE_chnl_stub     stub;    /* ctrl structure for stub */  
}TRACE_chnl;


/* Trace status manager - shared info */
typedef  struct
{
  TRACE_q_sh      buf;  /* Status buffer - shared */
}TRACE_stat_sh;

/* Trace status manager */
typedef  struct
{
  TRACE_q         buf;  /* Status buffer */
  TRACE_stat_sh*  sh;   /* Pointer to shared info */
}TRACE_stat;

/* Trace basic cnfg - shared info */
typedef struct
{
  TRACE_dev_cnfg      d_cnfg;      /* Device parameters         */
  TRACE_q_sh          buf;         /* Trace ring buffer - shared*/
  pthread_mutex_t     lrgmsg_lock; /* Mutex lock for large msg */
}TRACE_dev_sh;


/* Trace basic configuration */
typedef struct
{
  TRACE_q         buf;         /* Trace buffer                 */
  TRACE_dev_sh*   sh;          /* Pointer to shared info       */
  TRACE_IN_MSG*   in_msg;      /* Command buf for Trace-in msg */
}TRACE_dev;

typedef struct ATTRIBUTE((__packed__))
{
  U16 clas;
  U8 lvl;
}TRACE_proxy_file_cl;

typedef TRACE_proxy_file_cl TRACE_clasgrp;

typedef struct ATTRIBUTE((__packed__))
{
  /* PRQA: Lint Message 46: Fix from U32 to U8 is done as part of a hot fix */
  /*lint -save -e46 */
  U8 lvl_1  :4;  /* Lo-Nibble level info of class n     */
  U8 lvl_2  :4;  /* High-Nibble level info of class n+1 */
  /*lint -restore */
}TRACE_cfg_cl;

/* Proxy info - shared */
typedef struct
{
  U32             bfile_sz;
  U32             stor_med_sz;
  U32             max_bfile_cnt;
  U32             cur_bfile_cnt;
  U32             cur_sz;  
  BOOL            flush_mode;
  TRACE_PROXY_HDR hdr;
  BOOL            lookup_filecnt;
  TRACE_q_sh      buf;
  S8    cfg_file_path[TRACE_STORAGE_DEV_LEN + TRACE_FILE_NM_LEN];
  S8    bin_file_path[TRACE_STORAGE_DEV_LEN + TRACE_FILE_NM_LEN];
  U8   timestamp_mode;        /* timestamp flag, as per cfg  */
  U8   timestamp_flg;         /*timestamp flag, as per usr(current session)*/
  BOOL status;
  BOOL config_status;  /* Status of proxy task used for updating when TR_GET_TRACE_SETTINGS command from TTFis */
}TRACE_prxy_sh;

/* Proxy info */
typedef struct
{
  TRACE_q        buf;/*Proxy buffer */
  TRACE_prxy_sh* sh; /* Pointer to shared info */
  U8*            cl;  /* proxy class info */
  /* Proxy file nm - used for constructing filenames locally */
  S8    file_nm[TRACE_STORAGE_DEV_LEN + TRACE_FILE_NM_LEN];
  BOOL exit;
  S32             media_attach;

  U32             filesz;/* Proxy max file size */
  U32             max_logsz; /* Proxy Max log size */
  U8              blk_mode; /* Proxy block mode */
}TRACE_prxy;

/* Status flag for modules */
/* Changed from U8 to U32 as a part of Lint Error removal - agv2kor*/
typedef struct
{
  U32 core_actv    :1;       /* is core initialized    */
  U32 trc_actv     :1;       /* is comm io initialized */
  U32 svc_actv     :1;       /* is svc registered      */
  U32 chnl_actv    :1;       /* is channel initialized */
  U32 pxy_actv     :1;       /* is proxy initialized   */
  U32 prof_active  :1;       /* is profiler registered */
}TRACE_stat_flg;


typedef struct
{
  S32              stat;   /* slot indicator */
  pid_t            prc_id;
  TRACE_CALLBACK   fp;
}TRACE_notify_inf;

/* app notification - shared info */
typedef struct 
{
  S32   max;
  S32   idx;
}TRACE_notify_ctrl_sh;

/* app notification */
typedef struct
{
  TRACE_notify_ctrl_sh* sh;     /* pointer to shared info */
  TRACE_notify_inf*     inf;
  TRACE_notify_stub     stub;
}TRACE_notify_ctrl;

typedef struct
{
  U8 item[TRACE_MAX_DEVCNF_ITEM +1];
  U8 cnt;
}TRACE_devcnf_err;
/* Trace configuration parameters read from DEVCONF */
typedef struct
{
  S32       pri[TRACE_MAX_TASK];
  S32       stksz[TRACE_MAX_TASK];
  S32       lcid[TRACE_MAX_TASK];
  S32       prxy_flush_mode;
  size_t    prxy_log_sz;
  size_t    prxy_max_sz;
  S32       fixed_mem_start;
  S32       fixed_mem_end;
  S32       io_tmo;
  S32       cyclic_hdlr_tmo;
  S32       tcp_port_no;
  S32       flag_use_fixedmem;
  S32       flag_use_cyclic_op;
  S32       output_mode;
  S32       timestamp_mode;
  S32       trip_mode;
  S32       status_mode;
  S32       app_filter;
  U32       tcp_keepAlive;
  S8        tcp_ip_addr[L_IPADDR];
  S8        trc_cfg_loc[TRACE_DEVNM_LEN + 1];
  S8        script_file_loc[TRACE_DEVNM_LEN + 1];
  S8        script_file_nm[TRACE_FILE_NM_LEN + 1];
  S8        serial_dev_nm[TRACE_DEVNM_LEN + 1];
  S8        sim_dev_nm[TRACE_DEVNM_LEN + 1];
  S8        usb_h_dev_nm[TRACE_DEVNM_LEN + 1];
  S8        usb_f_dev_nm[TRACE_DEVNM_LEN + 1];
  S8        prxy_bin_loc[TRACE_DEVNM_LEN + 1];
  S8        prxy_cfg_loc[TRACE_DEVNM_LEN + 1];
  S8        chnl_skt_loc[TRACE_DEVNM_LEN + 1]; /* SWGIII-7042*/
  S8        chnl_skt_ntfy_loc[TRACE_DEVNM_LEN + 1];
  S8        tmpfilesd_cfg_path[TRACE_DEVNM_LEN + 1];
  size_t    trace_q_sz;
  size_t    stat_q_sz;
  size_t    chan_q_sz;
  size_t    prxy_q_sz;
  size_t    lrg_pkt_sz;
  TRACE_devcnf_err item_failed;
}TRACE_cfg;

typedef struct ATTRIBUTE((__packed__))
{
  U8  buf_mode;
  U8  io_chan;
  U8  tmstamp_mode;
  U8  trip_mode;
  U8  stat_mode;
  U8  app_filter;
  U8  rsvd[10];
  /* for bkward compatibility reasons
   * do not anything beyond this
   */
}TRACE_save_cnfg;
/*jov1kor-CRQ.No-XXXX- Implementation for Trace Internal Commands*/
typedef struct
{
  U8				  cmd_buf[TR_SEND_CMD_MSG_SZ];			
  U32				  buf_len;
  U8				  cmd_cmplt;
  U8				  reserved[3];
  pthread_mutex_t                 lock;
}TRACE_send_cmd_sh;

/* though this structure is meant for dualOS(specifically)
 * it is maintained as part of standalone mode also
 * purpose: to have common code base for all variants and
 * to avoid hellish #ifdef
 */
typedef struct
{
  S32   chn_id;
  U32   len;
  U32   reserved;   /* reserved varaiable */
}TRACE_chnl_route;

typedef struct
{
  TRACE_chnl_route  chnl_route;
  S32               trcCnfg_readStatus;/*SWGIIX-568, agv2kor, For storing read config status*/
  U32               versionCount;
  U32               reserved2;
  U32               reserved3;
}TRACE_ctrl_sh;

/*Function pointer which would be called from BE context, defined in FE context*/
typedef ER (*FEsaveCfg)(void);

/*Structure for storing Resource ID and tskspace info for access by BE*/
typedef struct
{
  FEsaveCfg          fp;
}TRACE_EarlyCfg_task_info;
/* Trace manager - shared info */
typedef struct _TRACE_mgr_sh
{
  size_t              shm_size;                 /* For stroing size of the shm, *
                                                 * whoever creates first shall update*/ //agv2kor
  TRACE_flg	      evt_id[EVT_MAX];          /* Union pointer for storing flag structures */
  TRACE_chnl_sh       chan;                     /* Channel buffer info      */
  TRACE_stat_sh       stat;                     /* Status buffer info       */
  TRACE_dev_sh        trc;                      /* Trace buffer info        */
  TRACE_prxy_sh       prxy;                     /* Proxy info               */
  TRACE_usr_cnfg      usr_cnfg;                 /* Fix me: usecase?         */
  TRACE_notify_ctrl_sh notify[TRACE_MAX_NOTIFY_EVT];   /* notification info for events */
  TRACE_class_cnfg_sh c_cnfg;                   /* shared info on class level */
  TRACE_ctrl_sh       ctrl;                     /* shared ctrl info           */
  /*Structure for storing Resource ID and tskspace info for access by BE*/
  TRACE_EarlyCfg_task_info taskContext;
  TRACE_send_cmd_sh	  sndcmd;	/*jov1kor-CRQ.No-XXXX- Implementation for Trace Internal Commands*/
  sem_t               sem;
  size_t              wr_tsk_state;	/* RTC-479458 / NCG3D-119027 :- TRACE_wr_task stage state */
}TRACE_mgr_sh;


/* Trace manager */
typedef struct _TRACE_mgr
{
  TRACE_sharedmem     shm;                      /* shared mem structure     */
  TRACE_mgr_sh*       sh;                       /* Pointer to shared info   */  
  TRACE_class_cnfg*   c_cnfg;                   /* Class configuration info */  
  TRACE_chnl          chan;                     /* Channel buffer info      */
  TRACE_stat          stat;                     /* Status buffer info       */
  TRACE_dev           trc;                      /* Trace buffer info        */     
  TRACE_prxy          prxy;                     /* Proxy info               */
  TRACE_io_cnfg       io_cfg[TRACE_MAX_IO_DEV]; /* Trace IO manager         */
  TRACE_notify_ctrl   notify[TRACE_MAX_NOTIFY_EVT];   /* notification info for events */
  TRACE_flowctrl      flow_ctrl;
  TRACE_cfg*          devconf;                  /* ptr to devconf config */
  TRACE_ctrl          ctrl;                     /* other ctrl informations */
#ifdef TRACE_ALD_INTERFACE_ENABLED
  TRACE_ALD_Mgr      ald_mgr;                   /* Struc*/
#endif
}TRACE_mgr;

typedef ER (*TRACE_io_func)(TRACE_mgr* mgr, TRACE_q* curq);
typedef ER (*TRACE_io_ext)(TRACE_mgr* mgr, U8* buf, S32 rdsz, S32* asiz);
/*SWGIII-3621: TRACE static constructor for init() dangerous */
typedef ER (*volatile TRACE_start_fp)();
/* Trace IO function pointers */
typedef struct
{
  TRACE_io_func   io_opn[TRACE_MAX_IO_DEV];
  TRACE_io_func   io_cls[TRACE_MAX_IO_DEV];
  TRACE_io_func   io_rd[TRACE_MAX_IO_DEV];
  TRACE_io_func   io_wr[TRACE_MAX_IO_DEV];
  TRACE_io_ext	  io_rd_ext[TRACE_MAX_IO_DEV];
  ER (*io_wr_epilog)(TRACE_mgr* mgr);
  ER (*io_rd_epilog)(TRACE_mgr* mgr);
}TRACE_io_fp;

struct errmem_message {
        int type; /*enum errmem_entry_type type*/
        unsigned short length;
        unsigned char message[256];
};

void TRACE_errmem_s32IOWrite(char* buffer, unsigned short size);

/**
 * Get appfilter status based on flag & proxy status
 *
 * \parm mgr Pointer to Trace manager
 *
 * return true if successful
 */

Inline BOOL TRACE_get_appfilter_status(TRACE_mgr* mgr);
Inline BOOL TRACE_get_appfilter_status(TRACE_mgr* mgr)
{
  BOOL ret = FALSE;
  if((0 != mgr->sh->trc.d_cnfg.flgs.app_filter) ||
    (TRUE == mgr->sh->prxy.status))
  {
    ret = TRUE;
  }
  return ret;
}

/**
 * Enable trip mode
 *
 * \parm mgr Pointer to Trace manager
 *
 * return >0 if modified, otherwise 0
 */
Inline ER TRACE_ena_trip_mode(TRACE_mgr* mgr);
Inline ER TRACE_ena_trip_mode(TRACE_mgr* mgr)
{
  ER rc = E_OK;
  U8 val = mgr->trc.in_msg->cmd_pkt.payld.payld[TRACE_TRIP_OFFSET];
  if(val != mgr->sh->trc.d_cnfg.flgs.trip)
  {
    if((val == TRIP_REC) || (val == TRIP_REPLAY) || (val == NO_TRIP))
    {
      mgr->sh->trc.d_cnfg.flgs.trip = (TRACE_trip_mode)val;
      /* non-zero value indicates status change */
      rc = (TRIP_REC | TRIP_REPLAY | NO_TRIP);
    }
    else
    {
      /* FixMe: invalid trip status */
    }
  }
  return rc;
}

/**
 * Enable block mode
 *
 * \parm mgr Pointer to Trace manager
 *
 * return None
 */
Inline void TRACE_ena_blockmd(TRACE_mgr* mgr);
Inline void TRACE_ena_blockmd(TRACE_mgr* mgr)
{
	char errMsg[TRACE_ERR_MEM_BLK_MODE_STR_SIZE]={0};

  U8 val = mgr->trc.in_msg->cmd_pkt.payld.payld[TRACE_BLCKMODE_OFFSET];

  /* kbm2kor: Fix - Block mode value defined in TRT is different 
   * from what is used in sources. change the value accordingly
   * TRT is not changed for backward compatibility
   */
  val = (U8)((val == TRACE_BLOCK_MODE)? TRACE_NO_BLOCK_MODE : TRACE_BLOCK_MODE);

  if(val != mgr->sh->trc.buf.blck_mode)
  {
    mgr->sh->trc.buf.blck_mode         = (TRACE_Q_BLOCK_STAT)val;
    mgr->sh->trc.d_cnfg.flgs.blck_mode = (TRACE_Q_BLOCK_STAT)val;
	
	sprintf( errMsg,"[cmd] BLOCKMODE { TRACE: %d  PROXY : %d }",(TRACE_BLOCK_MODE == mgr->sh->trc.buf.blck_mode)? TRUE : FALSE,(TRACE_BLOCK_MODE == mgr->sh->prxy.buf.blck_mode)? TRUE : FALSE);
	TRACE_errmem_s32IOWrite(errMsg, strlen(errMsg));
  }
  if(TRACE_BLOCK_MODE != mgr->sh->trc.buf.blck_mode)/*agv2kor*/
  {
	TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "NO-BLOCK mode is enabled\n");
  }
  else
  {
    TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "BLOCK mode is enabled\n");
  }
}


/**
 * Enable Proxy block mode
 *
 * \parm mgr Pointer to Trace manager
 *
 * return None
 */
Inline void TRACE_ena_prxy_blockmd(TRACE_mgr* mgr);
Inline void TRACE_ena_prxy_blockmd(TRACE_mgr* mgr)
{
  U8 val = mgr->trc.in_msg->cmd_pkt.payld.payld[TRACE_PROXY_BLOCKMODE_OFFSET];

  val = (U8)((val == TRACE_BLOCK_MODE)? TRACE_NO_BLOCK_MODE : TRACE_BLOCK_MODE);
  /*SWGIII-6993-Fix:Block mode status updation in proxy in trace-dlt mode */
  mgr->prxy.blk_mode = (TRACE_Q_BLOCK_STAT)val;
}

/**
 * Enable timestamp
 *
 * \parm mgr Pointer to Trace manager
 *
 * return None
 */
Inline void TRACE_ena_timestmp(TRACE_mgr* mgr);
Inline void TRACE_ena_timestmp(TRACE_mgr* mgr)
{
  U8 val = mgr->trc.in_msg->cmd_pkt.payld.payld[TRACE_TIMESTAMP_OFFSET];

  if(val & TRACE_VALID_TIMESTAMP)
  {
    mgr->sh->trc.d_cnfg.flgs.timestamp = (TRACE_TIMESTAMP_MODE)val;
  }
  else
  {
    /* Fix me: Traceout Timestamp request */
  }
}

/**
 * Calculate checksum
 *
 * \parm payld     Pointer to payload
 * \parm sz        Data size on which checksum shall be calculated
 * \parm chksum    Pointer to checksum buffer
 * \parm val       Reference value for checksum calculation
 *
 * return None
 */
Inline void TRACE_calculate_checksum(U8* payld, U32 sz, U8* chksum,
                                     TRACE_checksum val);
Inline void TRACE_calculate_checksum(U8* payld, U32 sz, U8* chksum,
                                     TRACE_checksum val)
{
  U32 i = 0;
  U8 chk_base = (U8)val; /* For removal of Lint Error e64 */

  for(; i < sz; i++)
  {
     chk_base ^= payld[i];
  }
  *chksum = chk_base;
}

/**
 * Construct TX message frame
 *
 * \parm trace_pkt Pointer to trace packet
 * \parm payld     Pointer to payload
 * \parm wrsz      Pointer to variable holding byte count
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
Inline void TRACE_construct_frame(TRACE_OUT_PKT* trace_pkt, U8* payld,
                                  S32* wrsz);
Inline void TRACE_construct_frame(TRACE_OUT_PKT* trace_pkt, U8* payld,
                                  S32* wrsz)
{
  if((trace_pkt->hdr.len > 0) && (trace_pkt->hdr.len <= TRACE_FRAME_PAYLD_SZ))
  {
    memcpy(payld, &trace_pkt->hdr, trace_pkt->hdr.len);
    *wrsz += trace_pkt->hdr.len;
  }
}

/**
 * Get current file size.
 *
 * \parm fd File descripter
 *
 * return \li 0 if unsuccessful
 *               else file size.
 */
Inline ssize_t TRACE_get_file_sz(S32 fd);
Inline ssize_t TRACE_get_file_sz(S32 fd)
{
  ssize_t sz = (ssize_t)lseek(fd, 0, SEEK_END);
  if(sz >= 0)
  {
    (void)lseek(fd, 0, SEEK_SET);
  }
  else
  {
    sz = 0;
  }
  return sz;
}


/**
 * Pack Word into a byte stream in Big Endian(Motorola) format
 *
 * \parm val word.
 * \parm buf Pointer to a byte stream
 *
 * return None.
 */
Inline void TRACE_pack_32bit_in_bigendian(const U32 val, U8* buf);
Inline void TRACE_pack_32bit_in_bigendian(const U32 val, U8* buf)
{
  U32 mask = TRACE_BYTE_MASK;
  buf[TRACE_POS_0] = (U8)((val & (mask << TRACE_SHIFT_24bits))
                                 >> TRACE_SHIFT_24bits);
  buf[TRACE_POS_1] = (U8)((val & (mask << TRACE_SHIFT_16bits))
                                 >> TRACE_SHIFT_16bits);
  buf[TRACE_POS_2] = (U8)((val & (mask << TRACE_SHIFT_8bits))
                                 >> TRACE_SHIFT_8bits);
  buf[TRACE_POS_3] = (U8)(val & mask);
}

Inline void TRACE_pack_16bit_in_bigendian(const U16 val, U8* buf);
Inline void TRACE_pack_16bit_in_bigendian(const U16 val, U8* buf)
{
  U16 mask = TRACE_BYTE_MASK;
  buf[TRACE_POS_0] = (U8)((val & (mask << TRACE_SHIFT_8bits))
                                 >> TRACE_SHIFT_8bits);
  buf[TRACE_POS_1] = (U8)(val & mask);
}

/**
 * Get Word from a byte stream in Big Endian(Motorola) format
 *
 * \parm buf Pointer to a byte stream
 * \parm val Pointer to a word
 *
 * return None.
 */
Inline void TRACE_get_32bit_frm_bigendian(const U8* buf, U32* val);
Inline void TRACE_get_32bit_frm_bigendian(const U8* buf, U32* val)
{
  (*val) = (((U32)buf[0] << TRACE_SHIFT_24bits) |
            ((U32)buf[1] << TRACE_SHIFT_16bits) |
            ((U32)buf[2] << TRACE_SHIFT_8bits)  |
             buf[3]);
}

Inline void TRACE_get_16bit_frm_bigendian(const U8* buf, U16* val);
Inline void TRACE_get_16bit_frm_bigendian(const U8* buf, U16* val)
{
  (*val) = (((U16)buf[0] << TRACE_SHIFT_8bits) | buf[1]);
}

/* Forward declerations */

/* used for FE and BE */
IMPORT const TRACE_io_fp g_TRACE_fnctbl;
IMPORT TRACE_mgr* g_TRACE_mgr;
IMPORT void TRACE_STATLIB_uninit(void);
IMPORT void TRACE_cre_hdr(U32 len, U16 clas, U8 lvl, TRACE_OUT_HDR* hdr,
                          U8 msg_typ);
IMPORT ER TRACE_snd_status(TRACE_mgr* mgr, const U8* buf, U8 len);

/* used for BE common and various kinds */
IMPORT ER TRACE_snd_blkmode_stat(TRACE_mgr* mgr);
IMPORT void TRACE_ini_shtdwn(TRACE_mgr* mgr);
IMPORT ER TRACE_process_cmd(TRACE_mgr* mgr);
IMPORT void TRACE_wr_common(TRACE_mgr* mgr, TRACE_q* curq,
                                    U8* rawpkt, S32* wrsz);
IMPORT void TRACE_snd_ovrcnt_info(TRACE_mgr* mgr, TRACE_io_func fp,
                                                    U32 ovrcnt);
IMPORT ER TRACE_stop(TRACE_mgr** tmgr);
 /*  SWGIIX-2071: CommandLine parameter not recieved in trace be */
IMPORT ER TRACE_start(void); 
IMPORT ER TRACE_commandLine_param(S32 argc, char** argv);

IMPORT ER TRACE_cre_proxy_cfg(TRACE_mgr* mgr);

IMPORT ER TRACE_init_task(const TRACE_task task, TRACE_max_task index, TRACE_mgr* mgr);

#endif /*_TRACE_BASE_H_*/
